#include <hidef.h>      /* for EnableInterrupts macro */
#include "derivative.h" /* include peripheral declarations */
#include "ts_functions.h"
#include "comm.h"
#include "control_constants.h"
#include "EEEfunctions.h"

#ifdef __cplusplus
 extern "C"
#endif


char num_readings = MEDIAN_LEN;
    
byte byte_to_TX;

word XP_debounce;

byte Press_detect = No;
byte Previous_press_detected = No;
byte discard_pt = No;

word wait_interval;

void initialization(void);
void MCU_init(void);     


/**************************************************************
*    Function: main()
*    Parameters: none
*    Return: none
*
*   Main loop. While the pen is down, this routine will 
*   debounce the pen touch detect, read the coordinate values,
*   and send them to the host processor. While the pen is up,
*   the CPU enters a STOP state, and will not resume until
*   until a pen-down interrupt occurs.
*
*   The sample rate is determined by the "wait_interval" value.
*   The loop first digitizes the X, Y coordinates, then enters
*   an inner loop waiting for a timer to expire. During this 
*   time the touch state is debounced. When the timer exires, the
*   timer is reset and the pen data is sent to the host system.
*   Note that the timer is running while the data is processed 
*   and sent to the host, so it must be ensured that the time
*   to do this is shorter than the timeout (otherwise no
*   pen debouncing will occur). In practice, the only problem
*   situation arised when the sampling rate is set for maximum
*   (100 pts/sec) and the communication rate is minimum (9600 bps).
*   In this case, the timer value is extended to give an actual
*   sampling rate of a little over 70 pts/sec.
*
*   The communication interval and sampling rate can be set
*   at compile time, thru comm,h and control_constants.h, 
*   respecitvely. 
* 
*   The point data is sent in a 4-byte packet in the following
*   format:
*
*      Bit: | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |  
*   Byte 0: | 0 | 0 | 0 | 0 | R | 0 | A | M |
*   Byte 1: | X7| X6| X5| X4| X3| X2| X1| X0|
*   Byte 2: | Y7| Y6| Y5| Y4| Y3| Y2| Y1| Y0|
*   Byte 3: |Y11|Y10| Y9| Y8|X11|X10| X9| X8|
*
*   The first byte is considered the header. The fields are defined as 
*   follows:
*       M -- MinMax calibrated (0=No, 1=Yes), 
*            currently always 1
*       A -- LCD Alignment calibrated (0=No, 1=Yes)
*       R -- release (0=no release, 1=release detected)
*
*   The 10-bit Y coordinate data is in bits Y9-Y0, while the 10-bit 
*   X coordinate data is in bits X9-X0. 
***************************************************************/
void main(void) 
{
    
    initialization();                                   //initialize the uC
    
    PTBPE_PTBPE3 = 0;                                   //Disable pull-up on XP pin

    if (*EEE_cal_data != 0xFF) use_df_coeff = 0;        //If translation coefficients are stored in memory, use them
    else use_df_coeff = 1;

    ts_calibrate(1);                                    //calibrate the coefficients including new min/max values for the touchscreen
    
    for(;;) 
    {
        __RESET_WATCHDOG();                             //Reset the watchdog timer

         PTBD_PTBD6 = 0;                                 //Output toggle for debugging/scope timing
  
//..... read the point data
        PTBPE_PTBPE3 = 0;                               //Disable pull-up on XP pin.  Do not want enabled when taking A/D readings.
        xmeas = Get_Point(read_xcoor, num_readings);    //Get the raw x-coordinate value
        ymeas = Get_Point(read_ycoor, num_readings);    //Get the ray y-coordinate value
        PTBPE_PTBPE3 = 1;                               //Enable pull-up on XP pin.  Needed to debounce the Pen-Down condition.
        calc_xy();                                      //Translate the raw coordinates to display coordinates
 
        PTBD_PTBD6 = 1;                                 //Output toggle for debugging/scope timing

        wait_interval = TX_interval;                    //Normal point interval

//..... Continuously debounce until the timer expires.        
        XP_debounce = 0;
        while (wait_to_TX_clock < wait_interval)
        {
            __RESET_WATCHDOG();                         //Reset the watchdog timer while spinning in this loop
       
            if(XP_Input)                                //If pen-up, the internal pull-up will cause XP to be high
                XP_debounce = 0;                        //Any pen-up will reset the debounce count
            else if (XP_debounce < 0xFFFF)
                XP_debounce++;                          //Pen-down: increment debounce counter

//.....if we support incoming commands from host, process them now
#ifdef SERIAL_RX_SUPPORTED
            if (temp_buffer[0] != 0)                    //RX buffer has data from the host uP
                Process_RX_Data();                      
            if (need_to_recal)
            {
                ts_calibrate(0);                        //  A change has occurred and the coefficeints 
                need_to_recal = No;                     //  need to be re-calculated.  But do not
                                                        //  recalculate the min/max reading on the 
                                                        //  panel in case a press is still occurring
                                                        //  that would corrupt the values
            }
#endif
        }

        wait_to_TX_clock = 0;                           //Reset clock to restart timer

//..... Pen-Down Detected                     
        if(XP_debounce > debounce_count)                
        {                                               
            Press_detect = Yes;                         //Set Press Detect flag
            if(LCD_aligned) 
                xy_data_toTX[0] = TX_AD_TD;             //Set up the first TX byte to indicate press detect
            else 
                xy_data_toTX[0] = TX_UA_TD;
            xy_data_toTX[1] = (byte)(xdisplay);         //Set up the 1st TX byte
            xy_data_toTX[2] = (byte)(ydisplay);         //Set up the 2nd TX byte
            xy_data_toTX[3] = (byte)((((ydisplay>>8) & 0x0F) << 4) | ((xdisplay>>8) & 0x0F));   //Set up the 3rd TX byte

            if (!Previous_press_detected)
                discard_pt = Yes;                       //Throw away the first point, as the data is often invalid
                Previous_press_detected = Yes;          //Set previous detect flag
        }                                              
//..... Pen-UP Detected
        else                                            
        {
            if(Press_detect)                            //Was a press detected on the last pass thru?
            {                                        
                Press_detect = No;                      //Turn off press detect flag
                if(LCD_aligned) 
                    xy_data_toTX[0] = TX_AD_RD;         //Set up first TX byte to indicate Pen-up detected
                else                                    //Only the 1st byte is changed.  The actual coordinate
                    xy_data_toTX[0] = TX_UA_RD;         //data comes from the last pen-down reading.
            } 
        }  

//..... Transmit the data
        if(Previous_press_detected && !discard_pt)      //Don't want a false KBI to cause communication        
            transmit((uchar *)(&xy_data_toTX) );        //TX the 4 byte data buffer
     
        discard_pt = No;                                

//..... Pen-up so turn on KBI and enter STOP        
        if(!Press_detect)                               
        {
            while (SCIS1_TC == 0);                      //Wait until all data has been transmitted before going to sleep
            Previous_press_detected = No;               //Prevents false KBIs from causing a TX
            KBIPE_KBIPE7 = 1;                           //Change XP into a KBI pin
            KBISC_KBACK  = 1;                           //Make sure no false interrupts are present
            KBISC_KBIE   = 1;                           //Enable the KBI module 
#ifdef SERIAL_RX_SUPPORTED
            if(!stay_awake_for_RX){                     // Allow to go to STOP mode
              SCIS2_RXEDGIF = 1;                        //Make sure there are no false Interrupts on the RX edge interrupt
              SCIBDH_RXEDGIE = 1;                       //Enable RX interrupt from Sleep Mode
              _asm STOP; 
            }
#else                                                  
            _asm STOP;                                  //Go to Stop3 mode
#endif
        }           
    } 
}


/**************************************************************
*    Function: device_init()
*    Parameters: none
*    Return: none
*
*    This function iniitializes the uC.  MCU_init() is a function
*       automatically generated by CodeWarrior's 'Device Initialization'
*       (This tool should not be used as MCU_init has been modified.  Be 
*       aware that if it runs again, manual changes made in MCU_init may be lost.)
*       This tool doe not allow you to use a pin for two different modules.  The
*       tool is set to make XP a GPIO.  XP also needs to be used as a KBI pin when
*       the uC is in WAIT mode.  The following assigns set the registers for the
*       XP when it will be used as a KBI.  The pin is converted from GPIO to KBI
*       just before going to WAIT mode.  It is converted back to GPIO in the KBI
*       interrupt routine.  This function also configures the UART's BAUD rate.
*       Finally, it loads the routines into RAM that are needed for EEPROM emulation.
***************************************************************/

void initialization(void) 
{
  
    MCU_init();         //Call the automated initialization routine

    KBISC_KBIE = 0;     //Ensure KBI is disabled
    KBIES_KBEDG7 = 0;   //Interrupt on falling edge or low level
    KBISC_KBIMOD = 1;   //Interrupt on both falling edge and low level 

    // change baud rate if needed (MCU_init() sets it to 9600)
#ifdef SERIAL_BAUD_38400
    SCIBDH = 0x40;
    SCIBDL = 0x0D;
#endif
#ifdef SERIAL_BAUD_19200
    SCIBDH = 0x40;
    SCIBDL = 0x1A;
#endif
#ifdef SERIAL_BAUD_9600
    // nothing to do here (default set in MCU_init)
#endif
        

    // Cannot run from Flash while writing/erasing it.  The following call
    // moves the Flash control function to RAM 

    CopyFLASHfuncsToRAM((uchar*)FLASH_COPY_START, (uchar*)RAM_FUNCS_START, RAM_FUNCS_SIZE);  
}